/*
	Alt-RAM driver for Alan Hourihane's MonSTer
	RAM/flash/IDE card.
	
	Jo Even Skarstein, 2015.
	
	Developed with PureC 1.1/standard libraries.
*/

#include <tos.h>
#include <aes.h>
#include <stdio.h>
#include <string.h>
#include "rsc.h"

enum {false, true};
#define	_ALLOW_LIVE_RAMTEST	/* Allows RAM-test while alt-RAM is in use */

/*
   NB! This code reference PureC library specific
   variables and may not compile with other
   compilers/libraries without modifications.
   
   _GemParBlk - Gem Parameter Block.
   _PgmSize - from pcstart.o. Program size.
*/

/* Wrapper around appl_init, that also detects if
   running from the auto-folder or not. */
int Appl_Init(void)
{
	int a;

	_GemParBlk.global[0] = 0;
	a = appl_init();
	
	if (_GemParBlk.global[0] == 0) /* Unchanged -> No AES */
		return -1;
	
	return a;
}

struct SETTINGS
{
	char magic[14];
	unsigned int reg;  /* MonSTer configuration register settings */
	unsigned char cpu; /* Mega STE CPU speed and cache settings */
};

struct SETTINGS settings = { "Magic number", 8, 0 };

struct COOKIE
{
	unsigned long id, value;
};

#define COOKIEJAR ((struct COOKIE **)0x5a0L)
unsigned long get_cookie(unsigned long id)
{
	long superstack = Super(NULL);
	struct COOKIE *cookiejar = *COOKIEJAR;

	if (cookiejar == NULL)
	{
		Super((void *)superstack);
		return 0;
	}
		
	while (cookiejar->id)
	{
		if (cookiejar->id == id)
		{
			Super((void *)superstack);
			return cookiejar->value;
		}
		
		cookiejar++;
	}

	Super((void *)superstack);
	return 0;
}

int new_cookie(unsigned long id, unsigned long value)
{
	int i = 0, max_cookies = 0;
	long superstack = Super(NULL);
	struct COOKIE *cookiejar = *COOKIEJAR;
	
	if (cookiejar == NULL)
	{
		Cconws("There is no cookiejar.\n\r");
		Super((void *)superstack);
		return -1;
	}
	
	while (cookiejar[i].id) /* Go to end of jar */
		i++;
	
	max_cookies = (int) cookiejar[i].value;

	if (i < max_cookies)
	{
		cookiejar[i].id = id;
		cookiejar[i].value = value;
		cookiejar[i+1].id = 0L;
		cookiejar[i+1].value = max_cookies;
	}
	else
	{
		Cconws("Cookiejar is full. Could not install _FRB cookie.\n\r");
		Super((void *)superstack);
		return -1;
	}

	Super((void *)superstack);
	return 0;
}

long maddalt(long addr, long size); /* PureC Maddalt is broken */
void set_monster_reg(unsigned short reg);
unsigned short get_monster_reg(void);
int check_monster(void);
int get_firmware_version(void);

unsigned long altram_base = 0x400000L;
int firmware_version, monSTer_present;

#define IS_MSTE (get_cookie('_MCH') == 0x00010010UL)
#define MSTE_CPUSPEED *(volatile unsigned char *)0xff8e21L
#define MSTE_CACHE 0x01
#define MSTE_16MHZ 0x02

void altram_driver(void)
{
	static long frb[16*1024];
	int frb_installed = false;

	Cconws("MonSTer alt-RAM driver\n\rJo Even Skarstein 2015 ->\n\r");

	if (!monSTer_present)
	{
		Cconws("No MonSTer available.\n\rAlt-RAM not installed.\n\n\r");
		return;
	}

	/* MegaSTE CPU speed, cache and alt-RAM size. */
	if (IS_MSTE)
	{
		long superstack = Super(NULL);
		MSTE_CPUSPEED = settings.cpu;
		Super((void *)superstack);

		if (firmware_version == -1)
			Cconws("Warning: Obsolete MonSTer firmware\n\rversion. Please upgrade.\n\r");

		if (settings.reg == 8)
			Cconws("Warning: Max 6Mb alt-RAM available on Mega STE.\n\r");

		settings.reg = (settings.reg == 8) ? 6 : settings.reg;

		if (settings.cpu & MSTE_16MHZ)
			Cconws("CPU speed set to 16MHz\n\r");
		
		if (settings.cpu & MSTE_CACHE)
			Cconws("CPU cache is enabled\n\r");

		Cconws("\n");
	}

	/* Set up alt-RAM */
	if (settings.reg == 0)
	{
		Cconws("No alt-RAM installed.\n\rRun M_ALTRAM.PRG from desktop to configure.\n\r");
		return;
	}

	/* Enable alt-RAM */
	set_monster_reg(settings.reg);

	/* Create _FRB cookie */
	if (get_cookie('_FRB') == 0)
	{
		if (new_cookie('_FRB', (long)frb) == -1)
		{	/* Error */
			Cconws("Could not install _FRB cookie.\n\rPress a key to continue.\n\r");
			Cconin();
			return;
		}
		else
			frb_installed = true;
	}

	if (maddalt(altram_base, settings.reg * 0x100000L) == 0)
	{
		printf("%d Mb alt-RAM installed at address 0x%lx\n",
			settings.reg, altram_base);

		if (frb_installed)
			Ptermres(_PgmSize, 0);
		else
			return;
	}
	else
	{
		Cconws("This TOS version does not support alt-RAM\n\rPress a key to continue.\n\r");
		Cconin();
	}
}


/* Simple RAM-test. */
#ifdef ALLOW_LIVE_RAMTEST
long int_on(void);
long int_off(void);
#endif

void test_ram(int megs)
{
	unsigned long *p = (void *)altram_base;
	int error = false, reg = get_monster_reg() & 0x0f, i;
	int dx, dy, dw, dh;

	if (megs == 0)
		return;

#ifndef ALLOW_LIVE_RAMTEST
	if (reg) /* Alt-RAM already enabled! */
	{
		form_alert(1,"[3][Alt-RAM is enabled and may be |in use by other programs. |RAM test may crash the system! ][ Cancel ]");
		form_alert(1,"[0][Boot without alt-RAM driver to |test alt-RAM. ][ Ok ]");
		return;
	}
#endif

	if (megs < 8)
	{
		char msg[] = "[0][You will now test the |first n Mb of alt-RAM ][ Ok ]";
		msg[33] = megs + '0';
		form_alert(1, msg);
	}

#ifdef ALLOW_LIVE_RAMTEST
	if (megs > reg) /* Don't reduce RAM-size, only increase */
#endif
		set_monster_reg(megs);

	form_center(progress, &dx, &dy, &dw, &dh);
	form_dial(FMD_START, dx,dy,0,0, dx,dy,dw,dh);
	progress[PROGRESS].ob_width = 0;
	progress[TITLE].ob_spec.free_string[8] = megs + '0';
	objc_draw(progress, 0, 8, dx, dy, dw, dh);

	graf_mouse(2, NULL);
	for (i = 0; i < 1024/4 * megs; i++)
	{
		static unsigned long a;
		int j;

#ifdef ALLOW_LIVE_RAMTEST
		int_off();
#endif
		for (j = 0;j < 1024; j++)
		{
			a = *p;
			*p = 0x5555AAAAL;
			if (*p != 0x5555AAAAL)
				error = true;
			
			*p = 0xAAAA5555L;
			if (*p != 0xAAAA5555L)
				error = true;
	
			*p = a;
			p++;

			if (error)
				break;
		}
#ifdef ALLOW_LIVE_RAMTEST
		int_on();
#endif

		if (error)
			break;

		if (i % megs == 0)
		{
			progress[PROGRESS].ob_width = i/megs;
			objc_draw(progress, PROGRESS, 8, dx, dy, dw, dh);
		}
	}

	form_dial(FMD_FINISH, dx,dy,0,0, dx,dy,dw,dh);
	graf_mouse(0, NULL);
	set_monster_reg(reg);
	
	if (error)
	{
		char msg[64];
		sprintf(msg, "[3][Error at address 0x%lx. |Please try again. ][ Ok ]",
			altram_base+(i*4));
		
		form_alert(1, msg);
	}
	else
		form_alert(1, "[0][Alt-RAM is OK ][ Good ]");
}

#define FNAME "m_altram.prg"

void update_settings(unsigned short reg, unsigned char cpu)
{
	long f = Fopen(FNAME, FO_RW);
	
	if (f > -1)
	{
		long i = 0, size = Fseek(0, (int) f, 2);
		struct SETTINGS *s = NULL;
		unsigned char *buf = NULL;

		Fseek(0, (int) f, 0);
		buf = Malloc(size);
		Fread((int) f, size, buf);
		Fclose((int) i);
		
		while (!s && i < size - sizeof(struct SETTINGS))
		{
			s = (struct SETTINGS *) strstr(&buf[i], settings.magic);
			i++;
		}

		if (s)
		{
			f = Fopen(FNAME, FO_WRITE);
			s->reg = reg;
			s->cpu = cpu;
			i = Fwrite((int) f, size, buf);
		}

		Fclose((int) f);
	}
	else
		form_alert(1, "[1][Could not find \"m_altram.prg\" ][ Ok ]");
}

OBJECT *init_dialog_tree(void)
{
	int i, rez = Getrez();

	for (i = 0; i < 13; i++)
		rsrc_obfix(dialog, i);

	for (i = 0; i < 4; i++)
		rsrc_obfix(progress, i);
		
	if (IS_MSTE)
	{
		dialog[CPU_16MHZ].ob_flags &= ~HIDETREE;
		dialog[CPU_16MHZ].ob_state |= (settings.cpu & MSTE_16MHZ ? SELECTED : 0);
		dialog[CPU_CACHE].ob_flags &= ~HIDETREE;
		dialog[CPU_CACHE].ob_state |= (settings.cpu & MSTE_CACHE ? SELECTED : 0);

		/* 8Mb not possible on Mega STE due to address conflict with VME bus */
		dialog[ALT_8MB].ob_flags |= HIDETREE;
	}

	dialog[ALT_0MB].ob_state |= (settings.reg == 0 ? SELECTED : 0);
	dialog[ALT_2MB].ob_state |= (settings.reg == 2 ? SELECTED : 0);
	dialog[ALT_4MB].ob_state |= (settings.reg == 4 ? SELECTED : 0);
	dialog[ALT_6MB].ob_state |= (settings.reg == 6 ? SELECTED : 0);
	if (!(dialog[ALT_8MB].ob_flags & HIDETREE))
		dialog[ALT_8MB].ob_state |= (settings.reg == 8 ? SELECTED : 0);

	if (rez == 0 || rez == 1)
		dialog[LOGO_HI].ob_flags |= HIDETREE;
	else
		dialog[LOGO_MID].ob_flags |= HIDETREE;

	if (!monSTer_present)
		dialog[TESTRAM].ob_state |= DISABLED;

	return dialog;
}

int main(void)
{
	OBJECT *d;
	int apid = Appl_Init();

	/* Check hardware */
	long superstack = Super(NULL);
	firmware_version = get_firmware_version();
	monSTer_present = check_monster();
	Super((void *)superstack);

#if 0
	printf("Firmware version: %d\n", firmware_version);
	Cconin();
	return 0;
#endif

	if (apid == -1)
	{
		/* Install alt-RAM if running from AUTO */
		altram_driver(); /* Will ptermres if success */
		return -1; /* Goes here only if altram_driver()
					  doesn't Ptermres.  */
	}

	d = init_dialog_tree();
	graf_mouse(0, 0L);

	if (IS_MSTE && firmware_version == -1)
		form_alert(1, "[3][MonSTer firmware obsolete. ][ Ok ]");

	/* Handle form */
	{
		int dx, dy, dw, dh, exit, cont = true;
		
		form_center(d, &dx, &dy, &dw, &dh);
		form_dial(FMD_START, dx,dy,0,0, dx,dy,dw,dh);
		
		while (cont)
		{
			unsigned short reg;
			unsigned char cpu;

			objc_draw(d, 0, 8, dx, dy, dw, dh);
			exit = form_do(d, 0);
			reg =	(d[ALT_0MB].ob_state & SELECTED) * 0 |
					(d[ALT_2MB].ob_state & SELECTED) * 2 |
					(d[ALT_4MB].ob_state & SELECTED) * 4 |
					(d[ALT_6MB].ob_state & SELECTED) * 6 |
					(d[ALT_8MB].ob_state & SELECTED) * 8;
					
			cpu  = d[CPU_16MHZ].ob_state & SELECTED ? MSTE_16MHZ : 0;
			cpu  |= (cpu && d[CPU_CACHE].ob_state & SELECTED) ? MSTE_CACHE : 0;
			
			switch (exit)
			{
				case TESTRAM:
					test_ram(reg);
					d[TESTRAM].ob_state &= ~SELECTED;
					break;
				case SAVE:
				{
					update_settings(reg, cpu);
					form_alert(1,"[1][You must reboot for the |changes to take effect. ][ Ok ]");
				}
				case CANCEL:
					cont = false;
			}
		}
		
		form_dial(FMD_FINISH, 0,0,0,0,0,0,0,0);
	}

	appl_exit();
	return 0;
}
