/*

	MonSTer flash- and configuration tool.
	
	Jo Even Skarstein 2015
	joska@online.no / "joska" @ atari-forum.com

	CRC-routines from TOS 2.06, disassembled by P. Putnik.

	Developed with Interface and 100% standard Pure C 1.1
	
*/

#include <tos.h>
#include <stdio.h>
#include <string.h>
#include <ext.h>
#include <stdlib.h>
#include <aes.h>
#include <assert.h>
#include "m_flash.h"

enum {false, true};
enum {msgclear, msgscroll};

/* From flash.s */
void fixcrc(unsigned char *tosimg);
void reset(void);
void tos_erase(int which, long flashaddr);
void set_monster_reg(unsigned short reg);
unsigned short get_monster_reg(void);

/* ROM image definitions */
#define TOS1LEN		0x30000L
#define TOS2LEN		0x40000L
#define FLASHDISKLEN	0x80000L

struct ROMIMAGE
{
	union
	{
		unsigned char *img;
		SYSHDR *osheader;
	} data;

	char	fname[14],
			versionstr[5];
	long	addr,
			size1;
	short	sectors,
			valid,
			version;
};

/*
   Functions to deal with the actual flash-ROM programming
*/

#define F0AAAA *(volatile unsigned short *)0xf0aaaaL
#define F05554 *(volatile unsigned short *)0xf05554L

unsigned long flasharea = 0xd00000L;

/* Mount the "other side" of the flash to address "flasharea" */
void enable_flash(void)
{
	unsigned short r = get_monster_reg() & 0xff0f;
	unsigned long f = flasharea >> 16;
	r |= (unsigned short)f;
	set_monster_reg(r);
}

/* Unmount writeable flash */
void disable_flash(void)
{
	set_monster_reg(get_monster_reg() & 0xff0f);
}

/* Unlocks flash, enables quicker writing */
void unlock_flash(void)
{
	F0AAAA = 0xAAAA;
	F05554 = 0x5555;
	F0AAAA = 0x2020;
	delay(10);
}

/* Writeprotect unlocked flash */
void lock_flash(void)
{
	volatile unsigned short *a = (unsigned short *)flasharea;

	*a = 0x9090;
	*a = 0x0000;
}

int compare(unsigned short *a, unsigned short *b, long count)
{
	long i;

	assert(a && b && (count > 0));

	for (i = 0; i < count; i++)
		if (a[i] != b[i])
			return true;

	return false;
}

int flash_sector(unsigned long sectoraddr, unsigned char *source)
{
	volatile unsigned short *a = (unsigned short *)flasharea;
	unsigned short *wsource = (unsigned short *)source;
	long i;
	unsigned long sector = sectoraddr/2;

	assert(source);
	
	/* NB! This requires the UNLOCK FLASH sequence to be run first */
	for (i = 0; i < 0x8000; i++)
	{
		a[sector + i] = 0xa0a0;
		a[sector + i] = wsource[i];
	}

	return compare(a + sector, wsource, 0x8000L);
}

/*
   Handle the UI
*/

OBJECT *dialog, *popups;

int do_popup(OBJECT *tree, int obj, int x, int y); /* popup.c */

char *rsc_string(int idx)
{
	char *t;
	static char e[] = "#ERR#";

	if (rsrc_gaddr(5, idx, &t))
		return t;
	
	return e;
}

long filesize(char *fname)
{
	long size, f = Fopen(fname, FO_READ);
	
	if (f > -1)
	{
		size = Fseek(0, (int) f, 2);
		Fclose((int) f);
		return size;
	}
	
	return f;
}

long read_file(char *fname, char *b, long len)
{
	long f, res;
	int handle;

	assert(fname && b && len > 0);

	f = Fopen(fname, FO_READ);
		
	if (f > -1)
	{
		handle = (int) f;
		res = Fread(handle, len, b);
		Fclose(handle);
		return res;
	}
	
	return f;
}

char *get_cwd(char *path, char *mask)
{
	short int drv = Dgetdrv();
	char tmp[256];

	assert(path && mask);

	if (Dgetpath(tmp, 0))
		return 0L; /* Error */
	else
	{
		sprintf(path, "%c:%s\\%s", drv + 'A', tmp, mask);
		path[0] = Dgetdrv() + 'A';
		path[1] = ':';
	}
	
	return path;
}

void progress(int p1, int p2) /* 0 - 100% */
{
	int absw = dialog[PROGRESS_BACK].ob_width, x, y;

	assert(p1 >= 0 && p1 <= 100 && p2 >= 0 && p2 <= 100);

	objc_offset(dialog, MAIN, &x, &y);
	dialog[PROGRESS_1].ob_width = (absw/100)*p1;
	dialog[PROGRESS_2].ob_width = (absw/100)*p2;
	objc_draw(dialog, PROGRESS_BACK, 8, x,y,
		dialog[MAIN].ob_width,dialog[MAIN].ob_height);
}

int load_image(struct ROMIMAGE *r)
{
	int button;
	static char path[128];
	char file[14];

	assert(r);

	get_cwd(path, "*.*");					

	if (fsel_input(path, file, &button) != 0)
	{
		graf_mouse(0,0L);
		if (button == 1)
		{
			long t, fsize;
			char load[142];

			path[strlen(path) - 3] = '\0';
			sprintf(load, "%s%s", path, file);
			fsize = filesize(load);
			
			if (fsize != r->size1)
			{
				char msg[200];
				sprintf(msg, rsc_string(ERR_SIZE), file, r->size1);
				form_alert(1, msg);
				return false;
			}

			if (r->data.img == NULL)
			{
				unsigned char *b = Malloc(r->size1);

				if (b)
					r->data.img = b;
				else
				{
					form_alert(1, rsc_string(NO_MEM));
					return false;
				}
			}
				
			t = read_file(load, r->data.img, r->size1);

			if (t < 0 || t != fsize)
			{
				char alert[128];
				sprintf(alert, rsc_string(NOLOAD), file);
				form_alert(1, alert);
				Mfree(r->data.img);
				r->data.img = NULL;
				return false;
			}
			
			strncpy(r->fname, file, sizeof(r->fname));
			r->version = 0;
			r->versionstr[0] = '\0';

			if (r->size1 != FLASHDISKLEN)
			{
				r->version = r->data.osheader->os_version;
				sprintf(r->versionstr, "%d.%.2d", (r->version >> 8), (r->version & 0xff));
			}

			r->valid = true;
			return true;
		}
	}

	return false;
}

char *set_dropdown_title(OBJECT *o, int i, char *t)
{
	char sym, *title;
	static int l = 0;
	
	assert(o && i >= 0 && t);
	title = o[i].ob_spec.tedinfo->te_ptext;

	/* Remember original string length */
	if (l == 0)
		l = (int) strlen(title);

	sym = title[0];
	strncpy(title, t, l);
	title[0] = sym;
	
	return title;
}

void message(char *str, int scroll)
{
	int x, y;
	static int w = 0;
	
	if (!w) w = (int) strlen(dialog[L1].ob_spec.free_string);

	assert(str);

	if (scroll)
	{
		strcpy(dialog[L1].ob_spec.free_string, dialog[L2].ob_spec.free_string);
		strcpy(dialog[L2].ob_spec.free_string, dialog[L3].ob_spec.free_string);
		strcpy(dialog[L3].ob_spec.free_string, dialog[L4].ob_spec.free_string);
		strcpy(dialog[L4].ob_spec.free_string, dialog[L5].ob_spec.free_string);
	}
	else /* Clear */
	{
		*dialog[L1].ob_spec.free_string = '\0';
		*dialog[L2].ob_spec.free_string = '\0';
		*dialog[L3].ob_spec.free_string = '\0';
		*dialog[L4].ob_spec.free_string = '\0';
	}

	strncpy(dialog[L5].ob_spec.free_string, str, w);
	objc_offset(dialog, STATUSBOX, &x, &y);
	objc_draw(dialog, STATUSBOX, 8, x, y, dialog[STATUSBOX].ob_width, dialog[STATUSBOX].ob_height);

	return;
}

void buffer_info(struct ROMIMAGE *r)
{
	char msg[20];

	message("TOS image:", msgclear);
	message(r->fname, msgscroll);

	if (r->version)
	{
		sprintf(msg, "Version: %s", r->versionstr);
		message(msg, msgscroll);
	}

	sprintf(msg, "Size: %ld", r->size1);
	message(msg, msgscroll);
	message("", msgscroll);
}

int main(void)
{
	int apid = appl_init(),
		 d_x, d_y, d_w, d_h, cont = true,
		 exit, rez = Getrez();

	struct ROMIMAGE romimages[3] = 
	{
		{ NULL, "", "", 0xc0000L, TOS1LEN,			3, false, 0 },
		{ NULL, "", "", 0x40000L, FLASHDISKLEN,	8, false, 0 },
		{ NULL, "", "", 0x00000L, TOS2LEN, 			4, false, 0 }
	};

	/* Initialise AES app */
	if (apid < 0)
	{
		printf("Could not init application. Exit.\n");
		return 1;
	}

	graf_mouse(0, 0L);

	if (rsrc_load("m_flash.rsc") == 0)
	{
		form_alert(1, "[3][Could not load monster.rsc][ Ok ]");
		goto exit;
	}

	rsrc_gaddr(R_TREE, POPUPS, &popups);
	rsrc_gaddr(R_TREE, MAIN, &dialog);

	/* Detect alt-RAM size to limit flash area address options */
	{
		unsigned int r = get_monster_reg() & 0x0f;
		
		if (r >= 6) /* 0x400000 - 0x9fffff */
		{
			popups[P_FADDR_80].ob_state |= DISABLED;
			popups[P_FADDR_90].ob_state |= DISABLED;
		}

		if (r == 8) /* 0x400000 - 0xbfffff */
		{
			popups[P_FADDR_A0].ob_state |= DISABLED;
			popups[P_FADDR_B0].ob_state |= DISABLED;
		}
	}

	/* ST LOW and MID needs a half height logo */
	dialog[LOGO_MID].ob_y = dialog[LOGO_MONO].ob_y;
	if (rez == 0 || rez == 1)
		dialog[LOGO_MONO].ob_flags |= HIDETREE;
	else
		dialog[LOGO_MID].ob_flags |= HIDETREE;

	/* Handle dialog */
	form_center(dialog, &d_x, &d_y, &d_w, &d_h);
	form_dial(FMD_START, d_x,d_y,0,0,d_x,d_y,d_w,d_h);

	message("", msgclear);
	progress(0, 0);
	
	while (cont)
	{
		objc_draw(dialog, 0, 8, d_x, d_y, d_w, d_h);
		exit = form_do(dialog, 0);
		
		switch (exit)
		{
			case HELP:
				break;

			case QUIT:
				cont = false;
				break;

			case RESET:
				if (form_alert(1,rsc_string(W_RESET)) == 2)
					reset();
				break;

			case TOS2X:
			case TOS1X:
			case FLASHDISK:
			{
				int x, y, m;
				struct ROMIMAGE *r;

				switch (exit)
				{
					case TOS1X:
						r = &romimages[0];
						break;
					case FLASHDISK:
						r = &romimages[1];
						break;
					case TOS2X:
						r = &romimages[2];
						break;
					default:
						assert(false);
						break;
				}

				if (r->valid)
				{
					popups[P_INFO].ob_state &= ~DISABLED;
					
					if (r->version == 0x0206)
						popups[P_CRC].ob_state &= ~DISABLED;
				}
				else
				{
					popups[P_CRC].ob_state |= DISABLED;
					popups[P_INFO].ob_state |= DISABLED;
				}
				
				objc_offset(dialog, exit, &x, &y);
				m = do_popup(popups, P_BUFFER, x, y);

				switch (m)
				{
					case P_LOAD:
						if (load_image(r))
						{
							char str[15];

							sprintf(str, "  %s", r->fname);
							set_dropdown_title(dialog, exit, str);
							buffer_info(r);
						}
						break;
					case P_NOFLASH:
					{
						char *t = popups[P_NOFLASH].ob_spec.free_string;

						set_dropdown_title(dialog, exit, t);
						r->valid = false;
						Mfree(r->data.img);
						r->data.img = NULL;
						message("", msgclear);
						break;
					}

					case P_INFO:
						buffer_info(r);
						break;

					case P_CRC:
						graf_mouse(2, 0L);
						fixcrc(r->data.img);
						graf_mouse(0, 0L);
						form_alert(1,rsc_string(CRCFIXED));
						break;
				}

				if (romimages[0].valid || romimages[1].valid || romimages[2].valid)
					dialog[FLASH].ob_state &= ~DISABLED;
				else
					dialog[FLASH].ob_state |= DISABLED;

				break;
			}

			case SET_FLASHADDR:
			{
				int x, y, m;
				
				objc_offset(dialog, SET_FLASHADDR, &x, &y);
				m = do_popup(popups, P_FLASHADDR, x, y);

				if (m != -1)
				{
					char *t = popups[m].ob_spec.free_string;
					set_dropdown_title(dialog, exit, t);
				}

				switch (m)
				{
					case P_FADDR_80:
						flasharea = 0x800000L;
						break;
					case P_FADDR_90:
						flasharea = 0x900000L;
						break;
					case P_FADDR_A0:
						flasharea = 0xA00000L;
						break;
					case P_FADDR_B0:
						flasharea = 0xB00000L;
						break;
					case P_FADDR_C0:
						flasharea = 0xC00000L;
						break;
					case P_FADDR_D0:
						flasharea = 0xD00000L;
						break;
				}

				break;
			}

			case FLASH:
			{
				int error = false, i,
					current_sector = 1,
					total_sectors = (romimages[0].valid ? romimages[0].sectors : 0) +
										(romimages[1].valid ? romimages[1].sectors : 0) +
										(romimages[2].valid ? romimages[2].sectors : 0);

				if (form_alert(1, rsc_string(WARNING)) == 1)
					break;
				
				graf_mouse(2, 0L);
				enable_flash();

				message("Flashing...", msgclear);
				for (i = 0; i < 3; i++)
				{
					if (romimages[i].valid)
					{
						int j;
						long ssp;
						
						message(romimages[i].fname, msgscroll);
						
						/* Erase flash */
						ssp = Super(0L);
						tos_erase(i, flasharea);
						Super((void *)ssp);

						/* Write image(s) to flash */
						unlock_flash();
						for (j = 0; j < romimages[i].sectors; j++)
						{
							progress( ((j+1) * 100)/romimages[i].sectors, 
										(current_sector++ * 100)/total_sectors );

							error += flash_sector(romimages[i].addr + (j * 0x10000L),
											romimages[i].data.img + (j * 0x10000L));
						}
						lock_flash();

						if (error)
						{
							char msg[128];
							int a;
							
							sprintf(msg, rsc_string(ERR_FLASH), romimages[i].fname);
							a = form_alert(1, msg);
							
							if (a == 1)
								break;
							
							if (a == 2)
							{
								/* Retry */
								current_sector -= romimages[i].sectors;
								i--;
								continue;
							}
						}
					}
				}

				disable_flash();
				graf_mouse(0, 0L);
				progress(0, 0);
				message("Done", msgscroll);

				if (!error)
					form_alert(1, rsc_string(OK_FLASH));
				
				break;
			}
		}

		dialog[exit].ob_state &= ~SELECTED;
	}
	
	form_dial(FMD_FINISH, 0,0,0,0,0,0,0,0);
	
exit:
	appl_exit();
	return 0;
}
